/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXNodeQuery.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx.master;

import org.snmp4j.agent.*;
import org.snmp4j.agent.request.SubRequest;
import org.snmp4j.smi.OctetString;

/**
 * The {@link AgentXNodeQuery} extends the {@link DefaultMOQuery} to allow on-the-fly query modifications and special
 * {@link #matchesQuery(ManagedObject)} support for explicitly match or not match {@link AgentXNode}s only.
 * @author Frank Fock
 */
public class AgentXNodeQuery extends DefaultMOQuery {

    /**
     * Query {@link AgentXNode}s only.
     */
    public static final int QUERY_AGENTX_NODES = 1;
    /**
     * Query non-{@link AgentXNode}s only.
     */
    public static final int QUERY_NON_AGENTX_NODES = 0;
    /**
     * Query all {@link ManagedObject}s.
     */
    public static final int QUERY_ALL = 2;

    private final int queryMode;

    /**
     * Creates a {@link AgentXNodeQuery} with query mode.
     * @param context
     *    the SNMPv3 context for the query.
     * @param scope
     *    the scope of the query.
     * @param queryMode
     *    an integer value specifying the search mode: {@link #QUERY_NON_AGENTX_NODES}..{@link #QUERY_ALL}.
     */
    public AgentXNodeQuery(OctetString context, MOScope scope, int queryMode) {
        super(new DefaultMOContextScope(context, scope), false);
        this.queryMode = queryMode;
    }

    /**
     * Returns its scope as mutable, to allow query modification on-the-fly.
     * @return
     *    the internal {@link DefaultMOContextScope} for modification.
     */
    public DefaultMOContextScope getMutableScope() {
        return (DefaultMOContextScope) getScope();
    }

    /**
     * Check if the provided {@link ManagedObject} matches this query type.
     * @param managedObject
     *    a {@link ManagedObject}
     * @return
     *   {@code true} if {@code managedObject} is matching the query mode {@link #QUERY_NON_AGENTX_NODES} or
     *   {@link #QUERY_AGENTX_NODES}. If mode is {@link #QUERY_ALL}, {@code true} is returned always.
     */
    public boolean matchesQuery(ManagedObject<?> managedObject) {
        if (managedObject instanceof AgentXNode) {
            return queryMode > QUERY_NON_AGENTX_NODES;
        }
        return queryMode != QUERY_AGENTX_NODES;
    }

    /**
     * Subtract the given scope from the query scope, i.e. reducing the scope by already done regions.
     * @param scope
     *    the scope to subtract.
     */
    public void subtractScope(MOScope scope) {
        getMutableScope().subtractScope(scope);
    }
}
