/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXMasterSession.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx.master;

import org.snmp4j.agent.agentx.AgentXProtocol;
import org.snmp4j.agent.agentx.AgentXSession;
import org.snmp4j.smi.Address;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;

import java.util.Map;
import java.util.HashMap;

/**
 * The {@code AgentXMasterSession} extends the base session
 * {@link AgentXSession} by agent capability information and
 * sysObjectID as well as sysDescr.
 *
 * @author Frank Fock
 * @version 1.0
 */
public class AgentXMasterSession<A extends Address> extends AgentXSession<A> {

    private static final long serialVersionUID = 2076735780892800483L;

    private int agentXVersion = AgentXProtocol.VERSION_1_0 & 0xFF;
    private final AgentXQueue queue;
    private final OID oid;
    private final OctetString descr;
    private Map<OID, OID> agentCaps;

    /**
     * Creates an new {@link AgentXMasterSession}.
     * @param sessionID
     *    the session ID.
     * @param queue
     *    the master agentx queue.
     * @param oid
     *    the OID identifying the master agent session.
     * @param descr
     *    the description of the master agent.
     */
    public AgentXMasterSession(int sessionID, AgentXQueue queue, OID oid, OctetString descr) {
        super(sessionID);
        this.queue = queue;
        this.oid = oid;
        this.descr = descr;
    }

    /**
     * Gets the AgentX request queue of this master agent.
     * @return
     *    the queue.
     */
    public AgentXQueue getQueue() {
        return queue;
    }

    /**
     * Gets the session OID.
     * @return
     *    an OID.
     */
    public OID getOid() {
        return oid;
    }

    /**
     * Gets the session description.
     * @return
     *    an {@link OctetString}.
     */
    public OctetString getDescr() {
        return descr;
    }

    /**
     * Gets the session's AgentX version.
     * @return
     *    a version identifies.
     */
    public int getAgentXVersion() {
        return agentXVersion;
    }

    /**
     * Sets the AgentX version identifier.
     * @param agentXVersion
     *    an AgentX version.
     */
    public void setAgentXVersion(int agentXVersion) {
        this.agentXVersion = agentXVersion;
    }

    /**
     * Add an agent caps entry.
     * @param sysORID
     *    the sysORID value of the new entry.
     * @param agentCapsIndex
     *    the agentXCapsIndex.
     */
    public synchronized void addAgentCaps(OID sysORID, OID agentCapsIndex) {
        if (agentCaps == null) {
            agentCaps = new HashMap<OID, OID>(10);
        }
        agentCaps.put(sysORID, agentCapsIndex);
    }

    /**
     * Remove an agent caps entry.
     * @param sysORID
     *    the sysORID value of the new entry.
     * @return
     *    the agentXCapsIndex or {@code null} if the entry does not exist.
     */
    public synchronized OID removeAgentCaps(OID sysORID) {
        if (agentCaps != null) {
            return agentCaps.remove(sysORID);
        }
        return null;
    }

}
