/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXGetBulkPDU.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx;

import java.io.Serial;
import java.nio.ByteBuffer;
import java.io.IOException;
import org.snmp4j.agent.MOScope;
import org.snmp4j.smi.OctetString;

/**
 * The {@link AgentXGetBulkPDU} represents the agentx-GetBulk-PDU defined by RFC 2741.
 */
public class AgentXGetBulkPDU extends AgentXRequestPDU {

    @Serial
    private static final long serialVersionUID = -8406339539172636681L;

    /**
     * GETBULK maximum repetitions.
     */
    protected short maxRepetitions;
    /**
     * GETBULK non-repeaters.
     */
    protected short nonRepeaters;

    /**
     * Creates a {@link AgentXGetBulkPDU}.
     * @param context
     *    the SNMPv3 context.
     * @param maxRepetitions
     *    the maximum GETBULK repetitions.
     * @param nonRepeaters
     *    the GETBULK non-repeaters.
     * @param ranges
     *    the request OID search ranges.
     */
    public AgentXGetBulkPDU(OctetString context,
                            short maxRepetitions,
                            short nonRepeaters,
                            MOScope[] ranges) {
        super(AGENTX_GETBULK_PDU, context, ranges);
        this.maxRepetitions = maxRepetitions;
        this.nonRepeaters = nonRepeaters;
    }

    /**
     * Creates a {@link AgentXGetBulkPDU} from a {@link AgentXMessageHeader}.
     * @param header
     *    the message header.
     */
    public AgentXGetBulkPDU(AgentXMessageHeader header) {
        super(header);
        if (header.getType() != AGENTX_GETBULK_PDU) {
            throw new IllegalArgumentException();
        }
    }

    /**
     * Gets the maximum GETBULK repetitions.
     * @return
     *    the maximum repetitions.
     */
    public short getMaxRepetitions() {
        return maxRepetitions;
    }

    /**
     * Gets the non repeaters.
     * @return
     *    the non repeaters.
     */
    public short getNonRepeaters() {
        return nonRepeaters;
    }

    /**
     * Sets the maximum repetitions for the GETBULK.
     * @param maxRepetitions
     *    the maximum repetitions.
     */
    public void setMaxRepetitions(short maxRepetitions) {
        this.maxRepetitions = maxRepetitions;
    }

    /**
     * Sets the non repeaters (0 for no non-repeaters).
     * @param nonRepeaters
     *    the non repeaters.
     */
    public void setNonRepeaters(short nonRepeaters) {
        this.nonRepeaters = nonRepeaters;
    }

    @Override
    protected void decodeAfterContext(ByteBuffer buf, int length) throws IOException {
        nonRepeaters = buf.getShort();
        maxRepetitions = buf.getShort();
        super.decodeAfterContext(buf, length);
    }

    @Override
    protected void encodeAfterContext(ByteBuffer buf) {
        buf.putShort(nonRepeaters);
        buf.putShort(maxRepetitions);
        super.encodeAfterContext(buf);
    }

    @Override
    protected int getAfterContextLength() {
        return AgentXProtocol.AGENTX_INT_SIZE + super.getAfterContextLength();
    }

    @Override
    protected String toStringExtMembers() {
        return super.toStringExtMembers()+",nonRepeaters="+nonRepeaters+
                ",maxRepetitions="+maxRepetitions;
    }

}
