/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXAddAgentCapsPDU.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx;

import java.io.*;
import java.nio.*;

import org.snmp4j.smi.*;

/**
 * The {@link AgentXAddAgentCapsPDU} represents an agentx-AddAgentCaps-PDU from RFC 2741.
 * @author Frank Fock
 */
public class AgentXAddAgentCapsPDU extends AgentXContextPDU {

    @Serial
    private static final long serialVersionUID = 5612445856184490038L;

    private OID id;
    private OctetString descr;

    /**
     * Creates a {@link AgentXAddAgentCapsPDU} from a {@link AgentXMessageHeader}.
     * @param header
     *   the message header.
     */
    public AgentXAddAgentCapsPDU(AgentXMessageHeader header) {
        super(header);
        if (header.getType() != AGENTX_ADDAGENTCAPS_PDU) {
            throw new IllegalArgumentException();
        }
    }

    /**
     * Creates a {@link AgentXAddAgentCapsPDU} from context, oid, and description.
     * @param context
     *   the context.
     * @param id
     *   the ID.
     * @param descr
     *   the description.
     */
    public AgentXAddAgentCapsPDU(OctetString context, OID id, OctetString descr) {
        super(AGENTX_ADDAGENTCAPS_PDU, context);
        this.id = id;
        this.descr = descr;
    }

    /**
     * Decode this PDU after the end of the encoded context.
     * @param buf
     *    the buffer positioned at the end of the encoded context.
     * @param length
     *    the length of the PDU to be decoded.
     * @throws IOException
     *    if the buffer contains less data than expected.
     */
    public void decodeAfterContext(ByteBuffer buf, int length) throws IOException {
        id = new OID();
        AgentXProtocol.decodeOID(buf, id);
        descr = AgentXProtocol.decodeOctetString(buf);
    }

    /**
     * Encode this PDU after already encoded context (thus context will not be encoded by this operation).
     * @param buf
     *    the target buffer.
     */
    public void encodeAfterContext(ByteBuffer buf) {
        AgentXProtocol.encodeOID(buf, id, false);
        AgentXProtocol.encodeOctetString(buf, descr);
    }

    /**
     * Gets the encoded length after an already encoded context.
     * @return
     *    the BER encoded length of ID and description.
     */
    public int getAfterContextLength() {
        return AgentXProtocol.getOIDLength(id) +
                AgentXProtocol.getOctetStringLength(descr.length());
    }

    /**
     * Gets the description.
     * @return
     *    an {@link OctetString}.
     */
    public OctetString getDescr() {
        return descr;
    }

    /**
     * Gets the ID.
     * @return
     *   an {@link OID}.
     */
    public OID getId() {
        return id;
    }

    /**
     * Sets the description.
     * @param descr
     *    an {@link OctetString}.
     */
    public void setDescr(OctetString descr) {
        this.descr = descr;
    }

    /**
     * Sets the ID.
     * @param id
     *   a {@link OID}.
     */
    public void setId(OID id) {
        this.id = id;
    }
}
