/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AbstractAgentXPending.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx.master;

import org.snmp4j.agent.agentx.AgentXProtocol;
import org.snmp4j.agent.request.SnmpRequest;
import org.snmp4j.smi.Address;
import org.snmp4j.smi.OID;

import java.io.Serial;
import java.util.Date;
import java.util.function.Function;

/**
 * The {@link AbstractAgentXPending} implements an abstract class for pending AgentX requests.
 * @param <A>
 *     the address type.
 */
public abstract class AbstractAgentXPending<A extends Address> implements AgentXPending<A> {

    @Serial
    private static final long serialVersionUID = -7904568557569179078L;

    /**
     * The region entry of the affected region by this pending request.
     */
    protected AgentXRegEntry<A> registration;
    private final SnmpRequest request;
    private boolean pending = true;
    private long timestamp = 0L;
    private int timeout = AgentXProtocol.DEFAULT_TIMEOUT_SECONDS;

    /**
     * Creates a new pending request based on affected region and request.
     * @param registration
     *    the affected registration region.
     * @param request
     *    the pending request.
     */
    public AbstractAgentXPending(AgentXRegEntry<A> registration, SnmpRequest request) {
        this.registration = registration;
        this.request = request;
    }

    /**
     * Sets the timestamp to the current system time milliseconds.
     */
    public void updateTimestamp() {
        timestamp = System.currentTimeMillis();
    }

    /**
     * Gets the system time milliseconds when the response has been received.
     * @return
     *    the latest timestamp in current system time milliseconds.
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * Sets the timeout seconds.
     * @param timeoutSeconds
     *    timeout seconds.
     */
    public void setTimeout(int timeoutSeconds) {
        this.timeout = timeoutSeconds;
    }

    /**
     * Sets the pending flag.
     * @param pending
     *    {@code false} if the response for the request has been received or the request timed out.
     */
    public void setPending(boolean pending) {
        this.pending = pending;
    }

    /**
     * Gets the request timeout in seconds.
     * @return
     *    a timeout in seconds.
     */
    public int getTimeout() {
        return timeout;
    }

    /**
     * The registration region affected by this request.
     * @return
     *    the registration region affected by this request.
     */
    public AgentXRegEntry<A> getRegistration() {
        return registration;
    }

    /**
     * Checks if the request is still pending.
     * @return
     *    {@code true} if the request is still pending.
     */
    public boolean isPending() {
        return pending;
    }

    /**
     * Gets the request.
     * @return
     *    a {@link SnmpRequest}.
     */
    public SnmpRequest getRequest() {
        return request;
    }

    @Override
    public String toString() {
        return getClass().getName() + "[" + toStringMembers() + "]";
    }

    /**
     * Returns a string representation of the class members to be overridden by subclasses.
     * @return
     *    a string representation of the sub-class members.
     */
    protected final String toStringMembers() {
        return "registration=" + registration + ",request=" + request + ",pending=" +
                pending + ",timestamp=" + new Date(timestamp) + ",timeout=" + timeout;
    }
}
