/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - PingEvent.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx.event;

import java.util.EventObject;

import org.snmp4j.agent.agentx.AgentXResponsePDU;
import org.snmp4j.agent.agentx.AgentXSession;
import org.snmp4j.smi.Address;

/**
 * The {@code PingEvent} object describes an AgentX ping result.
 * @author Frank Fock
 * @version 3.1.0
 */
public class PingEvent<A extends Address> extends EventObject {

    private static final long serialVersionUID = -659743205138032769L;

    private final AgentXSession<A> session;
    private AgentXResponsePDU response;
    private boolean closeSession;
    private boolean resetSession;
    private Exception error;

    /**
     * Creates a new {@link PingEvent} from a session and a ping response.
     * @param source
     *    the event source.
     * @param pingedSession
     *    the pinged {@link AgentXSession}.
     * @param pingResponse
     *    the ping response.
     */
    public PingEvent(Object source, AgentXSession<A> pingedSession, AgentXResponsePDU pingResponse){
        super(source);
        this.session = pingedSession;
        this.response = pingResponse;
    }

    /**
     * Creates a new {@link PingEvent} from a session and a ping response.
     * @param source
     *    the event source.
     * @param pingedSession
     *    the pinged {@link AgentXSession}.
     * @param error
     *    an exception.
     */
    public PingEvent(Object source, AgentXSession<A> pingedSession, Exception error){
        super(source);
        this.session = pingedSession;
        this.error = error;
    }

    /**
     * Gets the ping response.
     * @return
     *    a {@link AgentXResponsePDU}.
     */
    public AgentXResponsePDU getResponse() {
        return response;
    }

    /**
     * Gets the {@link AgentXSession}.
     * @return
     *    a {@link AgentXSession}.
     */
    public AgentXSession<A> getSession() {
        return session;
    }

    /**
     * Checks if the session is closing.
     * @return
     *    {@code true} if the session is closed or closing (see {@link #setCloseSession(boolean)}.
     */
    public boolean isCloseSession() {
        return closeSession;
    }

    /**
     * Checks if the session is resetting.
     * @return
     *    {@code true} if the session is resetting or resetted (see {@link #setResetSession(boolean)}.
     */
    public boolean isResetSession() {
        return resetSession;
    }

    /**
     * Gets the exception that caused on error.
     * @return
     *    an exception.
     */
    public Exception getError() {
        return error;
    }

    /**
     * Request closing of the session.
     * @param closeSession
     *    {@code true} to request closing the session.
     */
    public void setCloseSession(boolean closeSession) {
        this.closeSession = closeSession;
    }

    /**
     * Request resetting of the session.
     * @param resetSession
     *    {@code true} to request a reset of the session.
     */
    public void setResetSession(boolean resetSession) {
        this.resetSession = resetSession;
    }

    @Override
    public String toString() {
        return getClass().getName()+"[session="+session+",response="+response+
                ",error="+error+
                ",closeSession="+closeSession+",resetSession="+resetSession+"]";
    }
}
