/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXRequestPDU.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx;

import java.io.IOException;
import java.io.Serial;
import java.nio.ByteBuffer;

import org.snmp4j.agent.MOScope;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;
import java.util.Arrays;
import java.util.function.Function;

/**
 * The {@link AgentXRequestPDU} represents the base class for all agentx-Get*Request-PDUs defined by RFC 2741.
 */
public abstract class AgentXRequestPDU extends AgentXContextPDU {

    @Serial
    private static final long serialVersionUID = 4463505479247648609L;

    /**
     * The request ranges.
     */
    protected MOScope[] ranges;

    /**
     * The next OID filter can filter OIDs returned by AgentX GetNext and GetBulk operations before they are returned
     * to the master agent.
     * @since 4.1.0
     */
    protected Function<OID, Boolean> nextOidFilter;

    /**
     * Creates a {@link AgentXRequestPDU} from payload values.
     * @param type
     *   the PDU type.
     * @param context
     *   the SNMPv3 context.
     */
    protected AgentXRequestPDU(byte type, OctetString context) {
        super(type, context);
    }

    /**
     * Creates a {@link AgentXRequestPDU} from payload values.
     * @param type
     *   the PDU type.
     * @param context
     *   the SNMPv3 context.
     * @param ranges
     *   the requested ranges.
     */
    protected AgentXRequestPDU(byte type, OctetString context, MOScope[] ranges) {
        super(type, context);
        this.ranges = ranges;
    }

    /**
     * Creates a {@link AgentXRequestPDU} from a {@link AgentXMessageHeader}.
     * @param header
     *    the message header.
     */
    protected AgentXRequestPDU(AgentXMessageHeader header) {
        super(header);
    }

    @Override
    protected void decodeAfterContext(ByteBuffer buf, int length)
            throws IOException
    {
        ranges = AgentXProtocol.decodeRanges(buf);
    }

    @Override
    protected void encodeAfterContext(ByteBuffer buf) {
        AgentXProtocol.encodeRanges(buf, ranges);
    }

    @Override
    protected int getAfterContextLength() {
        return AgentXProtocol.getRangesLength(ranges);
    }

    /**
     * Gets the size of the ranges array.
     * @return
     *    the number of ranges in this request.
     */
    public int size() {
        if (ranges != null) {
            return ranges.length;
        }
        return 0;
    }

    /**
     * Gets the ranges of this request.
     * @return
     *   an array of {@link MOScope} instances.
     */
    public MOScope[] getRanges() {
        return ranges;
    }

    @Override
    protected String toStringExtMembers() {
        return super.toStringExtMembers()+",ranges="+Arrays.asList(ranges);
    }

}
