/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXRegion.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx;

import org.snmp4j.agent.DefaultMOScope;
import org.snmp4j.smi.OID;

import java.io.Serializable;

/**
 * The {@link AgentXRegion} represents a MIB region which can be an entire MIB or a single object-instance.
 */
public class AgentXRegion extends DefaultMOScope implements Comparable<AgentXRegion>, Serializable {

    private static final long serialVersionUID = -6110407395132112108L;
    private boolean singleOID;
    private byte rangeSubID;

    /**
     * Creates a new {@link AgentXRegion} from lower and upper bound.
     * @param lowerBound
     *   the lower bound of the region (including).
     * @param upperBound
     *    the upper bound of the region (excluding).
     */
    public AgentXRegion(OID lowerBound, OID upperBound) {
        super(lowerBound, true, upperBound, false);
    }

    /**
     * Creates a copy of an {@link AgentXRegion}.
     * @param other
     *    the original region.
     */
    public AgentXRegion(AgentXRegion other) {
        super(other.getLowerBound(), other.isLowerIncluded(),
                other.getUpperBound(), other.isUpperIncluded());
        this.singleOID = other.singleOID;
        this.rangeSubID = other.rangeSubID;
    }

    /**
     * Gets the range sub-id (i.e. the "wildcard" sub-identifier index).
     * @return
     *    the range sub ID index.
     */
    public byte getRangeSubID() {
        return rangeSubID;
    }

    /**
     * Checks if the MIB region represents a single object-instance.
     * @return
     *    {@code true} if this region represents a single object-instance.
     */
    public boolean isSingleOID() {
        return singleOID;
    }

    /**
     * Sets the range sub-id (i.e. the "wildcard" sub-identifier index).
     * @param rangeSubID
     *    the range sub ID index.
     */
    public void setRangeSubID(byte rangeSubID) {
        this.rangeSubID = rangeSubID;
    }

    /**
     * Indicates if the MIB region represents a single object-instance.
     * @param singleOID
     *    {@code true} if this region represents a single object-instance.
     */
    public void setSingleOID(boolean singleOID) {
        this.singleOID = singleOID;
    }

    /**
     * Gets the upper bound of the region.
     * @return
     *    {@code 0} if this is not a range region, and the upper bound range-sub-identifier otherwise.
     */
    public int getUpperBoundSubID() {
        if (rangeSubID != 0) {
            return upperBound.get(rangeSubID-1);
        }
        return 0;
    }

    /**
     * Checks if this region is a object instance range.
     * @return
     *    {@code true} if {@link #getRangeSubID()} is greater than zero.
     */
    public boolean isRange() {
        return (rangeSubID > 0);
    }

    /**
     * Checks if there is at least a single {@link OID} contained in this MIB region.
     * @return
     *    {@code true} if upper bound is greater or equals lower bound.
     */
    public boolean isEmpty() {
        return (lowerBound.compareTo(upperBound) >= 0);
    }

    /**
     * Gets the lower bound sub-ID of a range.
     * @return
     *    {@code 0} if this is not a range region, and the lower bound range-sub-identifier otherwise.
     */
    public int getLowerBoundSubID() {
        if (rangeSubID != 0) {
            return lowerBound.get(rangeSubID-1);
        }
        return 0;
    }

    @Override
    public int compareTo(AgentXRegion other) {
        int c = lowerBound.compareTo(other.lowerBound);
        if (c == 0) {
            c = upperBound.compareTo(other.upperBound);
            if (c == 0) {
                c = rangeSubID - other.rangeSubID;
            }
        }
        return c;
    }

    @Override
    public String toString() {
        return getClass().getName()+"[lowerBound="+lowerBound+
                ",lowerIncluded="+lowerIncluded+
                ",upperBound="+upperBound+
                ",upperIncluded="+upperIncluded+
                ",rangeSubID="+rangeSubID+
                ",upperBoundSubID="+getUpperBoundSubID()+"]";
    }
}
