/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXOpenPDU.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx;

import java.io.*;
import java.nio.*;

import org.snmp4j.smi.*;

/**
 * The {@link AgentXOpenPDU} represents the agentx-CleanupSet-PDU defined by RFC 2741.
 */
public class AgentXOpenPDU extends AgentXPDU {

    private static final long serialVersionUID = -8128999107664220880L;

    /**
     * AgentX timeout in seconds.
     */
    protected byte timeout;
    /**
     * The sub-agent ID.
     */
    protected OID subagentID;
    /**
     * The subagent description.
     */
    protected OctetString subagentDescr;

    AgentXOpenPDU(AgentXMessageHeader header) {
        super(header);
        if (header.getType() != AGENTX_OPEN_PDU) {
            throw new IllegalArgumentException("Incompatible PDU type");
        }
    }

    /**
     * Creates a {@link AgentXOpenPDU}.
     * @param sessionID
     *    a session ID.
     * @param transactionID
     *    a transaction ID.
     * @param packetID
     *    a packet ID.
     * @param timeout
     *    the packet timeout.
     * @param subagentID
     *    the sub-agent ID
     * @param subagentDescr
     *    the sub-agent description.
     */
    public AgentXOpenPDU(int sessionID, int transactionID, int packetID,
                         byte timeout,
                         OID subagentID, OctetString subagentDescr) {
        super(AGENTX_OPEN_PDU, (byte)0, sessionID, transactionID, packetID);
        this.timeout = timeout;
        this.subagentID = subagentID;
        this.subagentDescr = subagentDescr;
    }

    @Override
    public void decodePayload(ByteBuffer buf, int length) throws IOException {
        timeout = buf.get();
        // reserved
        buf.get();
        buf.get();
        buf.get();

        subagentID = new OID();
        AgentXProtocol.decodeOID(buf, subagentID);
        subagentDescr = AgentXProtocol.decodeOctetString(buf);
    }

    /**
     * Gets the sub-agent description.
     * @return
     *    a secription.
     */
    public OctetString getSubagentDescr() {
        return subagentDescr;
    }

    /**
     * Gets the sub-agent ID.
     * @return
     *    a {@link OID}.
     */
    public OID getSubagentID() {
        return subagentID;
    }

    /**
     * Gets the session timeout.
     * @return
     *   the timeout in seconds.
     */
    public byte getTimeout() {
        return timeout;
    }

    /**
     * Gets the sub-agent description.
     * @param subagentDescr
     *    the sub-agent description.
     */
    public void setSubagentDescr(OctetString subagentDescr) {
        this.subagentDescr = subagentDescr;
    }

    /**
     * Sets the sub-agent ID.
     * @param subagentID
     *    the sub-agent ID.
     */
    public void setSubagentID(OID subagentID) {
        this.subagentID = subagentID;
    }

    /**
     * Sets the session timeout in seconds.
     * @param timeout
     *    the timeout in seconds.
     */
    public void setTimeout(byte timeout) {
        this.timeout = timeout;
    }

    @Override
    public int getPayloadLength() {
        return AgentXProtocol.AGENTX_INT_SIZE +
                AgentXProtocol.getOIDLength(subagentID) +
                AgentXProtocol.getOctetStringLength(subagentDescr.length());
    }

    @Override
    public void encodePayload(ByteBuffer buf) {
        buf.put(timeout);
        buf.put(new byte[] { 0,0,0 }); // reserved
        AgentXProtocol.encodeOID(buf, subagentID, false);
        AgentXProtocol.encodeOctetString(buf, subagentDescr);
    }

    @Override
    protected void beforeEncode() {
    }
}
