/*
 * Decompiled with CFR 0.152.
 */
package jetbrains.exodus.util;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.zip.GZIPOutputStream;
import jetbrains.exodus.ExodusException;
import jetbrains.exodus.backup.BackupBean;
import jetbrains.exodus.backup.BackupStrategy;
import jetbrains.exodus.backup.Backupable;
import jetbrains.exodus.backup.VirtualFileDescriptor;
import jetbrains.exodus.util.IOUtil;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressBackupUtil {
    private static final Logger logger = LoggerFactory.getLogger(CompressBackupUtil.class);

    private CompressBackupUtil() {
    }

    @NotNull
    public static File backup(@NotNull Backupable source, @NotNull File backupRoot, @Nullable String backupNamePrefix, boolean zip2) throws Exception {
        if (!backupRoot.exists() && !backupRoot.mkdirs()) {
            throw new IOException("Failed to create " + backupRoot.getAbsolutePath());
        }
        String fileName = zip2 ? CompressBackupUtil.getTimeStampedZipFileName() : CompressBackupUtil.getTimeStampedTarGzFileName();
        File backupFile = new File(backupRoot, backupNamePrefix == null ? fileName : backupNamePrefix + fileName);
        return CompressBackupUtil.backup(source, backupFile, zip2);
    }

    @NotNull
    public static File backup(@NotNull BackupBean backupBean) throws Exception {
        backupBean.setBackupStartTicks(System.currentTimeMillis());
        return CompressBackupUtil.backup(backupBean, new File(backupBean.getBackupPath()), backupBean.getBackupNamePrefix(), backupBean.getBackupToZip());
    }

    @NotNull
    public static File backup(@NotNull Backupable source, @NotNull File target, boolean zip2) throws Exception {
        if (target.exists()) {
            throw new IOException("Backup file already exists:" + target.getAbsolutePath());
        }
        BackupStrategy strategy = source.getBackupStrategy();
        strategy.beforeBackup();
        try (BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(target));){
            ArchiveOutputStream archive;
            if (zip2) {
                ZipArchiveOutputStream zipArchive = new ZipArchiveOutputStream(output);
                zipArchive.setLevel(9);
                archive = zipArchive;
            } else {
                archive = new TarArchiveOutputStream(new GZIPOutputStream(output));
            }
            try (TarArchiveOutputStream aos = archive;){
                for (VirtualFileDescriptor fd : strategy.getContents()) {
                    long fileSize;
                    if (strategy.isInterrupted()) {
                        break;
                    }
                    if (!fd.hasContent() || (fileSize = Math.min(fd.getFileSize(), strategy.acceptFile(fd))) <= 0L) continue;
                    CompressBackupUtil.archiveFile(aos, fd, fileSize);
                }
            }
            if (strategy.isInterrupted()) {
                logger.info("Backup interrupted, deleting \"" + target.getName() + "\"...");
                IOUtil.deleteFile(target);
            } else {
                logger.info("Backup file \"" + target.getName() + "\" created.");
            }
        }
        catch (Throwable t) {
            strategy.onError(t);
            throw ExodusException.toExodusException(t, "Backup failed");
        }
        finally {
            strategy.afterBackup();
        }
        return target;
    }

    @NotNull
    public static String getTimeStampedTarGzFileName() {
        StringBuilder builder = new StringBuilder(30);
        CompressBackupUtil.appendTimeStamp(builder);
        builder.append(".tar.gz");
        return builder.toString();
    }

    @NotNull
    public static String getTimeStampedZipFileName() {
        StringBuilder builder = new StringBuilder(30);
        CompressBackupUtil.appendTimeStamp(builder);
        builder.append(".zip");
        return builder.toString();
    }

    public static void tar(@NotNull File source, @NotNull File dest) throws IOException {
        if (!source.exists()) {
            throw new IllegalArgumentException("No source file or folder exists: " + source.getAbsolutePath());
        }
        if (dest.exists()) {
            throw new IllegalArgumentException("Destination refers to existing file or folder: " + dest.getAbsolutePath());
        }
        try (TarArchiveOutputStream tarOut = new TarArchiveOutputStream(new GZIPOutputStream((OutputStream)new BufferedOutputStream(new FileOutputStream(dest)), 4096));){
            CompressBackupUtil.doTar("", source, tarOut);
        }
        catch (IOException e) {
            IOUtil.deleteFile(dest);
            throw e;
        }
    }

    private static void doTar(String pathInArchive, File source, TarArchiveOutputStream tarOut) throws IOException {
        if (source.isDirectory()) {
            for (File file : IOUtil.listFiles(source)) {
                CompressBackupUtil.doTar(pathInArchive + source.getName() + File.separator, file, tarOut);
            }
        } else {
            CompressBackupUtil.archiveFile(tarOut, new BackupStrategy.FileDescriptor(source, pathInArchive), source.length());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void archiveFile(@NotNull ArchiveOutputStream out, @NotNull VirtualFileDescriptor source, long fileSize) throws IOException {
        ArchiveEntry entry;
        if (!source.hasContent()) {
            throw new IllegalArgumentException("Provided source is not a file: " + source.getPath());
        }
        if (out instanceof TarArchiveOutputStream) {
            entry = new TarArchiveEntry(source.getPath() + source.getName());
            ((TarArchiveEntry)entry).setSize(fileSize);
            ((TarArchiveEntry)entry).setModTime(source.getTimeStamp());
            out.putArchiveEntry(entry);
        } else if (out instanceof ZipArchiveOutputStream) {
            entry = new ZipArchiveEntry(source.getPath() + source.getName());
            ((ZipArchiveEntry)entry).setSize(fileSize);
            ((ZipArchiveEntry)entry).setTime(source.getTimeStamp());
            out.putArchiveEntry(entry);
        } else {
            throw new IOException("Unknown archive output stream");
        }
        InputStream input = source.getInputStream();
        try {
            IOUtil.copyStreams(input, fileSize, out, IOUtil.getBUFFER_ALLOCATOR());
        }
        finally {
            if (source.shouldCloseStream()) {
                input.close();
            }
        }
        out.closeArchiveEntry();
    }

    private static void appendTimeStamp(StringBuilder builder) {
        Calendar now = Calendar.getInstance();
        builder.append(now.get(1));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(2) + 1);
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(5));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(11));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(12));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(13));
    }

    private static void appendZeroPadded(StringBuilder builder, int value) {
        if (value < 10) {
            builder.append('0');
        }
        builder.append(value);
    }
}

