/*_############################################################################
  _## 
  _##  SNMP4J-AgentX - AgentXVariableBindingPDU.java  
  _## 
  _##  Copyright (C) 2005-2026  Frank Fock (SNMP4J.org)
  _##  
  _##  This program is free software; you can redistribute it and/or modify
  _##  it under the terms of the GNU General Public License version 2 as 
  _##  published by the Free Software Foundation.
  _##
  _##  This program is distributed in the hope that it will be useful,
  _##  but WITHOUT ANY WARRANTY; without even the implied warranty of
  _##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  _##  GNU General Public License for more details.
  _##
  _##  You should have received a copy of the GNU General Public License
  _##  along with this program; if not, write to the Free Software
  _##  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  _##  MA  02110-1301  USA
  _##  
  _##########################################################################*/

package org.snmp4j.agent.agentx;

import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.VariableBinding;
import java.io.IOException;
import java.io.Serial;
import java.nio.ByteBuffer;
import java.util.Arrays;

/**
 * The {@link AgentXVariableBindingPDU} is the base class for all AgentX PDUs with a list of {@link VariableBinding}s.
 */
public class AgentXVariableBindingPDU extends AgentXContextPDU {

    @Serial
    private static final long serialVersionUID = -2031250227490466821L;

    /**
     * The variable bindings in the PDU.
     */
    protected VariableBinding[] variableBindings;

    /**
     * Creates a {@link AgentXVariableBindingPDU} from its type.
     * @param type
     *    a AgentX PDU type.
     * @param context
     *    the context.
     * @param vbs
     *    an array of {@link VariableBinding} instances.
     */
    protected AgentXVariableBindingPDU(byte type, OctetString context,
                                       VariableBinding[] vbs) {
        super(type, context);
        this.variableBindings = vbs;
    }

    /**
     * Creates a {@link AgentXVariableBindingPDU} from a {@link AgentXMessageHeader}.
     * @param header
     *    the message header.
     */
    protected AgentXVariableBindingPDU(AgentXMessageHeader header) {
        super(header);
    }

    /**
     * Creates a {@link AgentXVariableBindingPDU} from its type.
     * @param type
     *    a AgentX PDU type.
     * @param flags
     *    a set of AgentX flags, see {@link AgentXProtocol#FLAG_ALLOCATE_INDEX} for example.
     * @param sessionID
     *    a session ID.
     * @param transactionID
     *    a transaction ID.
     * @param packetID
     *    a packet ID.
     */
    protected AgentXVariableBindingPDU(byte type, byte flags, int sessionID,
                                       int transactionID, int packetID) {
        super(type, flags, sessionID, transactionID, packetID);
    }

    @Override
    public void decodeAfterContext(ByteBuffer buf, int length) throws IOException {
        variableBindings = AgentXProtocol.decodeVariableBindings(buf);
    }

    @Override
    public void encodeAfterContext(ByteBuffer buf) {
        AgentXProtocol.encodeVaribleBindings(buf, variableBindings);
    }

    @Override
    public int getAfterContextLength() {
        return AgentXProtocol.getVariableBindingsLength(variableBindings);
    }

    /**
     * Gets the {@link VariableBinding}s array.
     * @return
     *    the {@link VariableBinding}s array.
     */
    public VariableBinding[] getVariableBindings() {
        return variableBindings;
    }

    /**
     * Sets the {@link VariableBinding}s array.
     * @param variableBindings
     *    a {@link VariableBinding}s array.
     */
    public void setVariableBindings(VariableBinding[] variableBindings) {
        this.variableBindings = variableBindings;
    }

    /**
     * Gets the size of the {@link VariableBinding}s array.
     * @return
     *    the number of variable bindings in this PDU.
     */
    public int size() {
        return variableBindings.length;
    }

    @Override
    public String toStringExtMembers() {
        return super.toStringExtMembers()+",variableBindings="+
                ((variableBindings == null) ? null : Arrays.asList(variableBindings));
    }
}
